/*
 * RERApplet.java
 *
 * Created on 2006-01-18, 11:20
 */

package ch.hslu.cm.rer;

import ch.hslu.cm.rer.diagram.EntityFigure;
import ch.hslu.cm.rer.diagram.RelationalERDiagram;
import ch.hslu.cm.rer.diagram.RelationshipFigure;
import ch.hslu.cm.rer.model.RelationalERModel;
import ch.hslu.cm.rer.model.Entity;
import ch.hslu.cm.rer.model.Relationship;
import ch.randelshofer.swingcss.*;
import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.net.*;
import java.io.*;
import java.util.*;
import javax.swing.*;
import javax.swing.text.html.*;
import org.jhotdraw.app.action.edit.DeleteAction;
import org.jhotdraw.app.action.edit.DuplicateAction;
import org.jhotdraw.app.action.edit.SelectAllAction;
import org.jhotdraw.draw.AttributeKey;
import org.jhotdraw.draw.AttributeKeys;
import org.jhotdraw.draw.DefaultDrawingEditor;
import org.jhotdraw.draw.Drawing;
import org.jhotdraw.draw.DrawingEditor;
import org.jhotdraw.draw.DrawingView;
import org.jhotdraw.draw.TextFigure;
import org.jhotdraw.draw.event.ToolAdapter;
import org.jhotdraw.draw.event.ToolEvent;
import org.jhotdraw.draw.event.ToolListener;
import org.jhotdraw.draw.tool.ConnectionTool;
import org.jhotdraw.draw.tool.CreationTool;
import org.jhotdraw.draw.tool.DelegationSelectionTool;
import org.jhotdraw.draw.tool.Tool;
import org.jhotdraw.gui.Worker;
import org.jhotdraw.undo.UndoRedoManager;
import org.jhotdraw.util.ResourceBundleUtil;
import org.jhotdraw.xml.*;
/**
 * Applet with an editor for Relational Entity-Relationship diagrams.
 *
 * @author  wrandels
 */
public class RERApplet extends JApplet {
    private final static String VERSION = "2.1.103";
    private final static String NAME = "ConceptModeler[Relational E-R]";
    
    private LinkedList<Action> drawingActions;
    private LinkedList<Action> selectionActions;
    private UndoRedoManager undoManager;
    protected RelationalERDiagram problemDiagram, solutionDiagram;
    protected ResourceBundleUtil labels;
    private DrawingEditor editor;
    protected StyleManager stylemanager;
    protected StyleSheet stylesheet;
    private JProgressBar progressBar;
    private JToolBar toolBar;
    
    private class ToolButtonListener implements ItemListener {
        private Tool tool;
        public ToolButtonListener(Tool t) {
            this.tool = t;
        }
        @Override
        public void itemStateChanged(ItemEvent evt) {
            if (evt.getStateChange() == ItemEvent.SELECTED) {
                editor.setTool(tool);
            }
        }
    }
    
    protected DrawingView getDrawingView() {
        return drawingView;
    }
    
    /**
     * Handles toggle buttons for the tools.
     */
    private class ToolHandler extends ToolAdapter {
        private JToggleButton defaultToolButton;
        public ToolHandler(JToggleButton defaultToolButton) {
            this.defaultToolButton = defaultToolButton;
        }
    @Override
        public void toolDone(ToolEvent event) {
            defaultToolButton.setSelected(true);
        }
    }
    
    @Override
    public String getParameter(String name) {
        try {
            return super.getParameter(name);
        } catch (NullPointerException e) {
            return null;
        }
    }
    
    /**
     * Initializes the applet RERApplet
     */
    @Override
    public void init() {
        labels = ResourceBundleUtil.getBundle("ch.hslu.cm.rer.Labels", getLocale());
        
        try {
            UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
        } catch (Throwable e) {
            // Do nothing.
            // If we can't set the desired look and feel, UIManager does
            // automaticaly the right thing for us.
        }
        
        stylemanager = new StyleManager();
        if (getParameter("stylesheet") != null) {
            BufferedReader in = null;
            StringBuilder stylesheetData = new StringBuilder();
            try  {
                URL stylesheetURL = new URL(getDocumentBase(), getParameter("stylesheet"));
                in = new BufferedReader(new InputStreamReader(stylesheetURL.openStream(),"UTF8"));
                for (String line = in.readLine(); line != null; line = in.readLine()) {
                    stylesheetData.append(line);
                    stylesheetData.append('\n');
                }
                stylesheet = new StyleSheet();
                stylesheet.loadRules(new StringReader(stylesheetData.toString()), stylesheetURL);
                CSSLoader loader = new CSSLoader();
                loader.load(new StringReader(stylesheetData.toString()), stylemanager);
            } catch (IOException e) {
                e.printStackTrace();
            } finally {
                try { if (in != null) in.close(); } catch (IOException e) {}
            }
        }
        
        
        Container c = getContentPane();
        c.setLayout(new GridBagLayout());
        GridBagConstraints gbc;
        JLabel label;
        /*
        JLabel label = new JLabel("Loading...");
        label.setFont(label.getFont().deriveFont(Font.PLAIN, 11));
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        c.add(label, gbc);
         */
        label = new JLabel(NAME +" "+ VERSION);
        label.setFont(label.getFont().deriveFont(Font.BOLD, 14));
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        c.add(label, gbc);
        progressBar = new JProgressBar(0,6);
        gbc = new GridBagConstraints();
        gbc.gridx = 0;
        gbc.insets = new Insets(4,0,0,0);
        c.add(progressBar, gbc);
        reset();
    }
    
    /**
     * Does additional initialisation steps.
     */
    protected void initAdditional() {
        
    }
    
    /**
     * Resets the diagram to its initial state.
     */
    public void reset() {
        new Worker() {
            @Override
            public Object construct() {
                progressBar.setValue(progressBar.getValue()+1);
                Object result = null;
                
                editor = new DefaultDrawingEditor();
                progressBar.setValue(progressBar.getValue()+1);
                
                InputStream in = null;
                try {
                    NanoXMLDOMInput domi = null;
                    if (getParameter("data") != null) {
                        domi = new NanoXMLDOMInput(new RERFactory(), new StringReader(getParameter("data")));
                    } else if (getParameter("datafile") != null) {
                        URL url = new URL(getDocumentBase(), getParameter("datafile"));
                        in = url.openConnection().getInputStream();
                        domi = new NanoXMLDOMInput(new RERFactory(), in);
                    }
                    progressBar.setValue(progressBar.getValue()+1);
                    if (domi != null) {
                        domi.openElement("ConceptModelerRER");
                        
                        String instructionsText;
                        if (domi.getElementCount("Instructions") > 0) {
                            domi.openElement("Instructions");
                            instructionsText = (String) domi.readObject();
                            domi.closeElement();
                        } else {
                            instructionsText = "";
                        }
                        RelationalERDiagram pDiagram;
                        if (domi.getElementCount("Problem") > 0) {
                            domi.openElement("Problem");
                            RelationalERModel pSimulation = (RelationalERModel) domi.readObject(0);
                            pDiagram = (RelationalERDiagram) domi.readObject(1);
                            domi.closeElement();
                        } else {
                            pDiagram = new RelationalERDiagram();
                        }
                        RelationalERDiagram sDiagram;
                        if (domi.getElementCount("Solution") > 0) {
                            domi.openElement("Solution");
                            RelationalERModel sSimulation = (RelationalERModel) domi.readObject(0);
                            sDiagram = (RelationalERDiagram) domi.readObject(1);
                            domi.closeElement();
                            domi.closeElement();
                        } else {
                            sDiagram = new RelationalERDiagram();
                        }
                        result = new Object[] {
                            instructionsText,
                            pDiagram,
                            sDiagram
                        };
                    }
                } catch (Throwable t) {
                    result = t;
                } finally {
                    try { if (in != null) in.close(); } catch (IOException e) {}
                }
                
                progressBar.setValue(progressBar.getValue()+1);
                undoManager = new UndoRedoManager();
                toolBar = new JToolBar();
                addDrawActionsTo(toolBar);
                
                progressBar.setValue(progressBar.getValue()+1);
                return result;
            }
            @Override
            public void finished() {
                //long start = System.currentTimeMillis();
                JComponent c = (JComponent) getContentPane();
                c.setLayout(new BorderLayout());
                c.removeAll();
                
                Object result = getValue();
                initComponents();
                mainPanel.add(toolBar, BorderLayout.SOUTH);
                
                if (result != null && (result instanceof Object[])) {
                    problemDiagram = (RelationalERDiagram) ((Object[]) result)[1];
                    solutionDiagram = (RelationalERDiagram) ((Object[]) result)[2];
                } else {
                    problemDiagram = new RelationalERDiagram();
                    solutionDiagram = new RelationalERDiagram();
                }
                editor.add(drawingView);
                problemDiagram.addUndoableEditListener(undoManager);
                //initTools();
                drawingView.setDrawing(problemDiagram);
                //addDrawActionsTo(toolBar);
                
                if (result != null && (result instanceof Throwable)) {
                    problemDiagram.add(new TextFigure(result.toString()));
                    ((Throwable) result).printStackTrace();
                }
                
                initAdditional();
                
                // XXX - Styling the applet takes 4 seconds! This is way
                // too slow.
                /*
                long mid4= System.currentTimeMillis();
                ((JComponent) RERApplet.this.getContentPane()).putClientProperty("css.class","applet");
                drawingView.getComponent().putClientProperty("css.class","drawing");
                scrollPane.putClientProperty("css.class","scrollpane");
                toolBar.putClientProperty("css.class","toolbar");
                toolBar.setOpaque(true);
                stylemanager.style(RERApplet.this);
                long mid5= System.currentTimeMillis();
                */
                c.revalidate();
                /*
                long end = System.currentTimeMillis();
                System.out.println("RERApplet reset:finished elapsed: "+(end-start)+" m1:"+(mid4-start)+" m5-re:"+(mid5-mid4)+" e:"+(end-mid5));
                 */
            }
        }.start();
    }
    
    public void addDrawActionsTo(JToolBar tb) {
        ResourceBundleUtil labels = ResourceBundleUtil.getBundle("ch.hslu.cm.rer.Labels");
        ResourceBundleUtil drawLabels = ResourceBundleUtil.getBundle("org.jhotdraw.draw.Labels");
        
        JToggleButton t;
        JButton button;
        Tool tool;
        ButtonGroup group;
        HashMap<AttributeKey,Object> attributes;
        RelationshipFigure rf;
        
        group = new ButtonGroup();
        // Selection tool
        Tool selectionTool = new DelegationSelectionTool(getDrawingActions(), getSelectionActions());
        editor.setTool(selectionTool);
        t = new JToggleButton();
        ToolHandler toolHandler = new ToolHandler(t);
        drawLabels.configureToolBarButton(t, "selectionTool");
        t.setSelected(true);
        t.addItemListener(
                new ToolButtonListener(selectionTool)
                );
        t.setFocusable(false);
        group.add(t);
        tb.add(t);
        tb.addSeparator();
        
        // AttributeKeys for the entities
        attributes = new HashMap<AttributeKey,Object>();
        attributes.put(AttributeKeys.FILL_COLOR, RelationalERDiagram.ENTITY_NAME_COMPARTMENT_COLOR);
        attributes.put(AttributeKeys.STROKE_COLOR, RelationalERDiagram.ENTITY_STROKE_COLOR);
        attributes.put(AttributeKeys.TEXT_COLOR, RelationalERDiagram.ENTITY_TEXT_COLOR);
        
        // Fundamental Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntityFundamental");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_FUNDAMENTAL), attributes);

        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // Associative Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntityAssociative");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_ASSOCIATIVE), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // Attributive Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntityAttributive");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_ATTRIBUTIVE), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
                progressBar.setValue(progressBar.getValue()+1);
        
        // Supertype Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntitySupertype");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_SUPERTYPE), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        // Subtype Entity
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createEntitySubtype");
        tool = new CreationTool(new EntityFigure(Entity.TYPE_SUBTYPE), attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        tb.addSeparator();
        
        // AttributeKeys for Relationships
        attributes = new HashMap<AttributeKey,Object>();
        attributes.put(AttributeKeys.FILL_COLOR, RelationalERDiagram.RELATION_FILL_COLOR);
        attributes.put(AttributeKeys.STROKE_COLOR, RelationalERDiagram.RELATION_STROKE_COLOR);
        
        // Relationship 1:1
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRelationship11");
        rf = new RelationshipFigure();
        rf.getModel().setStartCardinality(1);
        rf.getModel().setEndCardinality(1);
        tool = new ConnectionTool(rf, attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRelationshipM1");
        rf = new RelationshipFigure();
        rf.getModel().setStartCardinality(Relationship.MANY_CARDINALITY);
        rf.getModel().setEndCardinality(1);
        tool = new ConnectionTool(rf, attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRelationship1M");
        rf = new RelationshipFigure();
        rf.getModel().setStartCardinality(1);
        rf.getModel().setEndCardinality(Relationship.MANY_CARDINALITY);
        tool = new ConnectionTool(rf, attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        t = new JToggleButton();
        labels.configureToolBarButton(t, "createRelationshipMM");
        rf = new RelationshipFigure();
        rf.getModel().setStartCardinality(Relationship.MANY_CARDINALITY);
        rf.getModel().setEndCardinality(Relationship.MANY_CARDINALITY);
        tool = new ConnectionTool(rf, attributes);
        t.addItemListener(new ToolButtonListener(tool));
        t.setFocusable(false);
        tool.addToolListener(toolHandler);
        group.add(t);
        tb.add(t);
        
        tb.addSeparator();
        button = tb.add(undoManager.getUndoAction());
        button.setOpaque(false);
        button = tb.add(undoManager.getRedoAction());
        button.setOpaque(false);
    }
    protected Drawing getDrawing() {
        return problemDiagram;
    }
    
    @Override
    public String getAppletInfo() {
        return NAME+"\nVersion "+VERSION
                +"\n\nCopyright 2006-007 \u00a9 Lucerne University of Applied Sciences and Arts"
                +"\nAlle Rechte vorbehalten."
                +"\n\nDiese Software basiert auf"
                +"\nSoftBricks \u00a9 2007 Werner Randelshofer"
                +"\nJHotDraw 7 \u00a9 2007 JHotDraw.org"
                +"\nNanoXML \u00a9 2000-2002 Marc De Scheemaecker";
    }
    
    public void setData(String text) {
        problemDiagram.removeAllChildren();
        if (text != null && text.length() > 0) {
            StringReader in = new StringReader(text);
            try {
                undoManager.discardAllEdits();
                NanoXMLDOMInput domi = new NanoXMLDOMInput(new RERFactory(), in);
                domi.openElement("ConceptModelerRER");
                
                problemDiagram = (RelationalERDiagram) domi.readObject(0);
                drawingView.setDrawing(problemDiagram);
                problemDiagram.addUndoableEditListener(undoManager);
            } catch (Throwable e) {
                problemDiagram.removeAllChildren();
                TextFigure tf = new TextFigure();
                tf.setText(e.getMessage());
                tf.setBounds(new Point2D.Double(10,10), new Point2D.Double(100,100));
                problemDiagram.add(tf);
                e.printStackTrace();
            } finally {
                if (in != null) in.close();
            }
        }
    }
    public String getData() {
        CharArrayWriter out = new CharArrayWriter();
        try {
            NanoXMLDOMOutput domo = new NanoXMLDOMOutput(new RERFactory());
            domo.openElement("ConceptModelerRER");
            domo.writeObject(problemDiagram);
            domo.closeElement();
            domo.save(out);
        } catch (IOException e) {
            TextFigure tf = new TextFigure();
            tf.setText(e.getMessage());
            tf.setBounds(new Point2D.Double(10,10), new Point2D.Double(100,100));
            problemDiagram.add(tf);
            e.printStackTrace();
        } finally {
            if (out != null) out.close();
        }
        return out.toString();
    }
    private Collection<Action> getDrawingActions() {
        if (drawingActions == null) {
            drawingActions = new LinkedList<Action>();
            drawingActions.add(undoManager.getUndoAction());
            drawingActions.add(undoManager.getRedoAction());
            drawingActions.add(null); // separator
            drawingActions.add(new SelectAllAction());
        }
        return drawingActions;
    }
    private Collection<Action> getSelectionActions() {
        if (selectionActions == null) {
            selectionActions = new LinkedList<Action>();
            selectionActions.add(new DuplicateAction());
            selectionActions.add(new DeleteAction());
        }
        return selectionActions;
    }
    
    @Override
    public String[][] getParameterInfo() {
        return new String[][] {
            { "data", "String", "the data to be displayed by this applet." },
            { "datafile", "URL", "an URL to the data to be displayed by this applet." },
            { "stylesheet", "URL", "An URL to the CSS stylesheet for this applet." },
        };
    }
    
    public static void main(String[] args) {
        SwingUtilities.invokeLater(new Runnable() {
            @Override
            public void run() {
                JFrame f = new JFrame(NAME+" "+VERSION);
                f.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
                f.setSize(500,300);
                RERApplet applet = new RERApplet();
                f.getContentPane().add(applet);
                applet.init();
                f.setVisible(true);
                applet.start();
            }
        });
    }
    
    public void toolStarted(ToolEvent event) {
    }
    
    /** This method is called from within the init() method to
     * initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is
     * always regenerated by the Form Editor.
     */
    // <editor-fold defaultstate="collapsed" desc=" Generated Code ">//GEN-BEGIN:initComponents
    private void initComponents() {
        toolGroup = new javax.swing.ButtonGroup();
        mainPanel = new javax.swing.JPanel();
        scrollPane = new javax.swing.JScrollPane();
        drawingView = new org.jhotdraw.draw.DefaultDrawingView();

        mainPanel.setLayout(new java.awt.BorderLayout());

        scrollPane.setViewportView(drawingView);

        mainPanel.add(scrollPane, java.awt.BorderLayout.CENTER);

        getContentPane().add(mainPanel, java.awt.BorderLayout.CENTER);

    }// </editor-fold>//GEN-END:initComponents
    
    
    // Variables declaration - do not modify//GEN-BEGIN:variables
    private org.jhotdraw.draw.DefaultDrawingView drawingView;
    private javax.swing.JPanel mainPanel;
    private javax.swing.JScrollPane scrollPane;
    private javax.swing.ButtonGroup toolGroup;
    // End of variables declaration//GEN-END:variables
    
}
